const express = require("express");
const bodyParser = require("body-parser");
const mysql = require("mysql");
const cors = require("cors");
const multer = require("multer");
const path = require("path");

const app = express();
const port = process.env.PORT || 3001;
// Middleware
app.use(
  cors({
    origin: "*", // อนุญาตให้ทุกโดเมนเข้าถึง API ของคุณ
    methods: ["GET", "POST", "PUT", "DELETE"], // ระบุวิธีที่อนุญาต
  })
);
app.use(bodyParser.json());
app.use(bodyParser.urlencoded({ extended: true }));
app.use(express.urlencoded({ extended: true }));
// Multer setup for file uploads
const storage = multer.diskStorage({
  destination: (req, file, cb) => {
    cb(null, "uploads/bill"); // Ensure this directory exists
  },
  filename: (req, file, cb) => {
    // Generate a unique filename
    cb(null, `${Date.now()}_${file.originalname}`);
  },
});

const upload = multer({ storage: storage });

// MySQL Connection
// const db = mysql.createConnection({
//   host: "ps05",
//   user: "zilmapsz_pickup",
//   password: "25082541zZ?><",
//   database: "zilmapsz_pickupservice",
// });

const db = mysql.createConnection({
  host: "localhost",
  user: "root",
  password: "",
  database: "pickupservice",
});

db.connect((err) => {
  if (err) throw err;
  console.log("Connected to database");
});

app.post("/api/login", (req, res) => {
  const { username, password } = req.body;
  const query =
    "SELECT * FROM `ps_user` WHERE user_username = ? AND user_pass = ?";

  db.query(query, [username, password], (err, results) => {
    if (err) {
      console.error("Error executing query:", err);
      res.status(500).send("Server error");
      return;
    }

    if (results.length > 0) {
      const user = results[0]; // Get the user object from the results
      res.json({
        status: "ok",
        message: "Logged in",
        user: {
          id: user.user_id,
          username: user.user_username,
          status: user.user_status,
          address: user.user_address,
        },
      });
      console.log("User logged in:", user);
    } else {
      res.status(401).send("Invalid credentials");
    }
  });
});

app.post("/api/register", (req, res) => {
  const { username, password } = req.body;

  console.log(username);
  console.log(password);

  // ตรวจสอบว่ามีการกรอกชื่อผู้ใช้หรือรหัสผ่านหรือไม่
  if (!username || !password) {
    return res.status(400).json({
      status: "error",
      message: "ชื่อผู้ใช้และรหัสผ่านเป็นสิ่งที่จำเป็น",
    });
  }

  // ตรวจสอบว่ามีชื่อผู้ใช้นี้อยู่ในฐานข้อมูลแล้วหรือไม่
  const checkUserQuery = "SELECT * FROM `ps_user` WHERE user_username = ?";

  db.query(checkUserQuery, [username], (err, results) => {
    if (err) {
      console.error("ข้อผิดพลาดในการดำเนินการคำสั่งตรวจสอบผู้ใช้:", err);
      return res.status(500).json({
        status: "error",
        message: "ข้อผิดพลาดของเซิร์ฟเวอร์",
        error: err.message,
      });
    }

    // ถ้ามีชื่อผู้ใช้นี้อยู่แล้ว
    if (results.length > 0) {
      return res.status(400).json({
        status: "error",
        message: "ชื่อผู้ใช้นี้มีอยู่แล้ว",
      });
    }

    // ถ้าไม่มีชื่อผู้ใช้นี้อยู่ ให้ดำเนินการการบันทึกข้อมูลผู้ใช้ใหม่
    const insertQuery =
      "INSERT INTO `ps_user` (user_username, user_pass, user_status) VALUES (?, ?, ?)";

    console.log("กำลังดำเนินการคำสั่ง:", insertQuery);
    console.log("ค่า:", [username, password, 1]);

    db.query(insertQuery, [username, password, 1], (err, result) => {
      if (err) {
        console.error("ข้อผิดพลาดในการดำเนินการคำสั่งบันทึกผู้ใช้:", err);
        return res.status(500).json({
          status: "error",
          message: "ข้อผิดพลาดของเซิร์ฟเวอร์",
          error: err.message,
        });
      }

      // ตอบกลับด้วยข้อความสำเร็จหากผู้ใช้ถูกเพิ่ม
      res.json({
        status: "ok",
        message: "ลงทะเบียนผู้ใช้สำเร็จ",
        userId: result.insertId, // รับ ID ของผู้ใช้ที่เพิ่ม
      });
    });
  });
});

// Endpoint for image upload
app.post("/api/upload", upload.single("image"), (req, res) => {
  try {
    res.json({ imageUrl: `/uploads/${req.file.filename}` }); // Return the image URL
  } catch (error) {
    console.error("Error uploading image:", error);
    res.status(500).send("Error uploading image");
  }
});

// Existing routes
app.get("/api/package-details", (req, res) => {
  const query = "SELECT * FROM `package_detail`";

  db.query(query, (err, results) => {
    if (err) {
      console.error("Error executing query:", err);
      res.status(500).send("Server error");
      return;
    }

    res.json(results);
  });
});

// app.get("/api/user-table", (req, res) => {
//   const query =
//     "SELECT * FROM `user_pickup` INNER JOIN user_service where user_pickup.userv_id = user_service.userv_id;";

//   db.query(query, (err, results) => {
//     if (err) {
//       console.error("Error executing query:", err);
//       res.status(500).send("Server error");
//       return;
//     }

//     res.json(results);
//   });
// });

app.post("/api/update/useradd", upload.none(), (req, res) => {
  console.log("Request Body:", req.body); // Debugging: log the request body

  const { userv_id, user_id } = req.body;

  if (!userv_id || !user_id) {
    return res.status(400).json({ message: "Invalid input data" });
  }

  const sql = "UPDATE ps_user SET user_address = ? WHERE user_id = ?";
  const values = [userv_id, user_id];

  db.query(sql, values, (err, result) => {
    if (err) {
      console.error(err);
      return res.status(500).json({ message: "Database update error" });
    }
    if (result.affectedRows === 0) {
      return res.status(404).json({
        message: "Record not found",
        sql: sql,
        values: values,
      });
    }
    res.status(200).json({ message: "Record updated successfully" });
  });
});

app.get("/api/user-table", (req, res) => {
  const user_id = req.query.user_id; // Retrieve user_id from query parameters

  if (!user_id) {
    return res.status(400).send({
      code: 1,
      isSuccess: false,
      message: "User ID is required",
    });
  }

  const query = `
    SELECT *,
        up.*, 
        last_bill.last_bill_id, 
        last_cost.last_bill_cost
    FROM 
        user_pickup AS up
    INNER JOIN 
        user_service AS us ON up.userv_id = us.userv_id
    LEFT JOIN (
        SELECT 
            pb.pickup_id, 
            MAX(pb.bill_cost) AS last_bill_cost 
        FROM 
            ps_bill AS pb 
        INNER JOIN (
            SELECT 
                pickup_id, 
                MAX(bill_id) AS last_bill_id 
            FROM 
                ps_bill 
            GROUP BY 
                pickup_id
        ) AS last_bill ON pb.bill_id = last_bill.last_bill_id
        GROUP BY 
            pb.pickup_id
    ) AS last_cost ON up.pickup_id = last_cost.pickup_id
    LEFT JOIN (
        SELECT 
            pickup_id, 
            MAX(bill_id) AS last_bill_id 
        FROM 
            ps_bill 
        GROUP BY 
            pickup_id
    ) AS last_bill ON last_cost.pickup_id = last_bill.pickup_id
    WHERE 
        us.user_id = ?
    GROUP BY 
        up.pickup_id
    ORDER BY 
        up.userp_date DESC 

    `;

  db.query(query, [user_id], (err, results) => {
    if (err) {
      console.error("Error executing query:", err);
      res.status(500).send({
        code: 1,
        isSuccess: false,
        message: "Server error",
      });
      return;
    }

    res.json({
      code: 0,
      isSuccess: true,
      data: {
        rows: results.length,
        items: results,
      },
    });
  });
});

const today = new Date().toISOString().split("T")[0]; // Current date in ISO 8601 format (YYYY-MM-DD)

app.get("/api/employee-table", (req, res) => {
  const query = `
    SELECT * 
FROM user_pickup
INNER JOIN user_service ON user_pickup.userv_id = user_service.userv_id
WHERE user_pickup.userp_date <= '${today}' 
ORDER BY user_pickup.userp_date DESC;`;

  db.query(query, (err, results) => {
    if (err) {
      console.error("Error executing query:", err);
      res.status(500).send({
        code: 1,
        isSuccess: false,
        message: "Server error",
      });
      return;
    }

    res.json({
      code: 0,
      isSuccess: true,
      data: {
        rows: results.length,
        items: results,
      },
    });
  });
});

app.get("/api/pickup-table", (req, res) => {
  const query = `
    SELECT * 
    FROM user_pickup
    INNER JOIN user_service ON user_pickup.userv_id = user_service.userv_id
    WHERE user_pickup.userp_status = '1' OR user_pickup.userp_status = '2' ;
  `;

  db.query(query, (err, results) => {
    if (err) {
      console.error("Error executing query:", err);
      res.status(500).send({
        code: 1,
        isSuccess: false,
        message: "Server error",
      });
      return;
    }

    res.json({
      code: 0,
      isSuccess: true,
      data: {
        rows: results.length,
        items: results,
      },
    });
  });
});

app.post("/api/pickup", upload.single("userp_image"), (req, res) => {
  const { userv_id, userp_quantity, userp_date, userp_package, userp_status } =
    req.body;

  // Get the filename from the uploaded file
  const userp_image = req.file ? req.file.filename : null;

  const sql =
    "INSERT INTO user_pickup (userv_id, userp_date, userp_package, userp_quantity, userp_image, userp_status) VALUES (?, ?, ?, ?, ?, ?)";
  const values = [
    userv_id,
    userp_date,
    userp_package,
    userp_quantity,
    userp_image,
    userp_status,
  ];

  db.query(sql, values, (err, result) => {
    if (err) {
      console.error(err);
      res.status(500).send("Database insert error");
      return;
    }
    res.status(200).send("Data inserted successfully");
  });
});

app.post("/api/service", (req, res) => {
  const {
    user_id,
    userv_name,
    userv_tel,
    userv_letlong,
    userv_address,
    userv_detail,
  } = req.body;

  const sql =
    "INSERT INTO user_service (user_id, userv_name, userv_tel, userv_letlong, userv_address, userv_detail) VALUES (?, ?, ?, ?, ?, ?)";
  const values = [
    user_id,
    userv_name,
    userv_tel,
    userv_letlong,
    userv_address,
    userv_detail,
  ];

  db.query(sql, values, (err, result) => {
    if (err) {
      console.error(err);
      res.status(500).send("Database insert error");
      return;
    }
    const userv_id = result.insertId;
    res.status(200).json({ userv_id }); // Send userv_id back as JSON response
  });
});

app.post("/api/update/service", (req, res) => {
  const {
    userv_id, // ID ของแถวที่ต้องการอัปเดต
    userv_name,
    userv_tel,
    userv_letlong,
    userv_address,
    userv_detail,
  } = req.body;

  const sql =
    "UPDATE user_service SET userv_name = ?, userv_tel = ?, userv_letlong = ?, userv_address = ?, userv_detail = ? WHERE userv_id = ?";
  const values = [
    userv_name,
    userv_tel,
    userv_letlong,
    userv_address,
    userv_detail,
    userv_id, // ID ของแถวที่ต้องการอัปเดต
  ];

  db.query(sql, values, (err, result) => {
    if (err) {
      console.error(err);
      res.status(500).send("Database update error");
      return;
    }
    if (result.affectedRows === 0) {
      // ถ้าไม่มีแถวที่ถูกอัปเดต
      res.status(404).send("Record not found");
      return;
    }
    res.status(200).send("Record updated successfully");
  });
});

app.post("/api/update-status", (req, res) => {
  const uservId = req.body.modalName; // หรือ req.params.userv_id ขึ้นอยู่กับการส่งค่ามา
  const status = req.body.status;
  const pickupId = req.body.pickup_id;

  console.log("status ID:", status);
  console.log("Pickup ID:", pickupId);

  // Construct the query based on the presence of pickupId
  let query;
  const queryParams = [];

  query = "UPDATE `user_pickup` SET `userp_status` = ? WHERE pickup_id = ?";
  queryParams.push(status, pickupId);

  // Execute the update query
  db.query(query, queryParams, (err, results) => {
    if (err) {
      console.error("Error executing query:", err);
      res.status(500).send("Server error");
      return;
    }

    // Log the executed SQL query
    const executedQuery = db.format(query, queryParams);
    console.log("Executed SQL query:", executedQuery);

    // Query to retrieve updated data
    const selectQuery =
      pickupId != null
        ? "SELECT * FROM `user_pickup` WHERE pickup_id = ?"
        : "SELECT * FROM `user_pickup` WHERE userv_id = ?";
    const selectParams = pickupId != null ? [pickupId] : [uservId];

    db.query(selectQuery, selectParams, (err, updatedResults) => {
      if (err) {
        console.error("Error retrieving updated data:", err);
        res.status(500).send("Server error");
        return;
      }

      if (updatedResults.length > 0) {
        const updatedUserpStatus = updatedResults[0].userp_status;
        res.json({ userp_status: updatedUserpStatus });
      } else {
        res.status(404).send("Data not found");
      }
    });
  });
});

app.post("/api/employee/datapickup", (req, res) => {
  console.log("Request body:", req.body);
  const pickup_id = req.body.name;
  const query = `
    SELECT * 
FROM user_pickup
INNER JOIN user_service ON user_pickup.userv_id = user_service.userv_id
WHERE user_pickup.pickup_id = ?`;

  console.log("Executing query:", query);
  console.log("With parameters:", [pickup_id]);

  db.query(query, [pickup_id], (err, results) => {
    if (err) {
      console.error("Error executing query:", err);
      res.status(500).send({
        code: 1,
        isSuccess: false,
        message: "Server error",
      });
      return;
    }

    res.json({
      code: 0,
      isSuccess: true,
      data: {
        rows: results.length,
        items: results,
      },
    });
  });
});

app.post("/api/upload/bill", upload.single("file"), (req, res) => {
  const { pickup_id, bill_cost } = req.body;
  const bill_image = req.file ? req.file.filename : null; // Use filename instead of buffer

  const query = `
    INSERT INTO ps_bill (pickup_id, bill_image, bill_cost, bill_status, bill_cancel)
    VALUES (?, ?, ?, 'p', NULL)
  `;

  db.query(query, [pickup_id, bill_image, bill_cost], (err, results) => {
    if (err) {
      console.error("Error inserting data:", err);
      return res
        .status(500)
        .json({ success: false, message: "Error inserting data" });
    }
    res.json({ success: true, message: "Data inserted successfully" });
  });
});

app.post("/api/update/bill", upload.single("file"), (req, res) => {
  console.log("File info:", req.file);
  const { pickup_id, rejected, comment } = req.body;
  const bill_image = req.file ? req.file.filename : null;

  // Determine the query based on whether the bill is rejected or not
  let query;
  let queryParams;
  console.log(rejected);
  if (rejected === "true") {
    // SQL query for rejected bills
    query = `
      UPDATE ps_bill 
      SET bill_cancel = ?, bill_status = 'n'
      WHERE pickup_id = ?
    `;
    queryParams = [comment, pickup_id];
  } else {
    // SQL query for updating the bill image
    query = `
      UPDATE ps_bill 
      SET bill_payimage = ? , bill_status = 'p' , bill_cancel = NULL
      WHERE pickup_id = ?
    `;
    queryParams = [bill_image, pickup_id];
  }

  console.log("Executing SQL Query:", query);
  console.log("With Parameters:", queryParams);

  db.query(query, queryParams, (err, results) => {
    if (err) {
      console.error("Error updating data:", err);
      return res
        .status(500)
        .json({ success: false, message: "Error updating data" });
    }
    res.json({ success: true, message: "Data updated successfully" });
  });
});

app.get("/api/upload/showbill", (req, res) => {
  const { pickup_id } = req.query;
  console.log(pickup_id);
  const query = `
   SELECT * FROM ps_bill WHERE pickup_id = ? ORDER BY bill_id DESC LIMIT 1;

  `;

  db.query(query, [pickup_id], (err, results) => {
    if (err) {
      console.error("Error fetching data:", err);
      return res
        .status(500)
        .json({ success: false, message: "Error fetching data" });
    }
    res.json({ success: true, message: results });
  });
});

app.get("/api/show-bill/:filename", (req, res) => {
  const { filename } = req.params;
  const filePath = path.join(__dirname, "uploads/bill", filename);

  res.sendFile(filePath, (err) => {
    if (err) {
      console.error("Error sending file:", err);
      res.status(404).send("File not found");
    }
  });
});

app.get("/api/show-image/:filename", (req, res) => {
  const { filename } = req.params;
  const filePath = path.join(__dirname, "uploads/bill", filename);

  res.sendFile(filePath, (err) => {
    if (err) {
      console.error("Error sending file:", err);
      res.status(404).send("File not found");
    }
  });
});

app.get("/api/insert/address", (req, res) => {
  const { user_id, userv_id } = req.query;

  // ตรวจสอบว่ามี user_id และ userv_id หรือไม่
  let query = "SELECT * FROM user_service WHERE 1=1"; // ตั้งค่าเริ่มต้นที่เป็น true เพื่อช่วยในการประกอบ query
  let queryParams = [];

  if (user_id) {
    query += " AND user_id = ?";
    queryParams.push(user_id);
  }

  if (userv_id) {
    query += " AND userv_id = ?";
    queryParams.push(userv_id);
  }

  db.query(query, queryParams, (err, results) => {
    if (err) {
      console.error("Error fetching data:", err);
      return res
        .status(500)
        .json({ success: false, message: "Error fetching data" });
    }
    res.json({ success: true, message: results });
  });
});

app.post("/api/noti/insert", (req, res) => {
  const { noti_detail, user_id, noti_status } = req.body;

  const sql =
    "INSERT INTO user_pickup (noti_date, noti_detail, user_id, noti_status) VALUES (?, ?, ?, ?)";
  const values = [today, noti_detail, user_id, noti_status];

  db.query(sql, values, (err, result) => {
    if (err) {
      console.error(err);
      res.status(500).send("Database insert error");
      return;
    }
    res.status(200).send("Data inserted successfully");
  });
});

app.listen(port, () => {
  console.log(`Server running on port ${port}`);
});
